<?php
/**
 * Marquee functions and definitions
 *
 * @package Marquee
 */

/**
 * Set the content width based on the theme's design and stylesheet.
 */
if ( ! isset( $content_width ) ) {
	$content_width = 653; /* pixels */
}

if ( ! isset( $full_content_width ) ) {
	$full_content_width = 990; /* pixels */
}

if ( ! class_exists( 'ThemeBoy_Marquee' ) ) :

class ThemeBoy_Marquee {

	/**
	 * @var string
	 */
	public $version = '1.2';

	/**
	 * @var string
	 */
	public $slug = 'marquee';

	/**
	 * @var string
	 */
	public $name = 'Marquee';

	public function __construct() {
		// Define constants
		$this->define_constants();
		
		// Include plugins
		$this->include_plugins();

		// Hooks
		add_action( 'wp_enqueue_scripts', array( $this, 'scripts' ), 11 );
		add_filter( 'rookie_theme_starter_content', array( $this, 'theme_starter_content' ) );
		add_action( 'rookie_customize_register', array( $this, 'customize_register' ) );
		add_filter( 'rookie_footer_copyright', array( $this, 'footer_copyright' ), 20 );
		add_filter( 'rookie_footer_credit', array( $this, 'footer_credit' ), 20 );
		add_filter( 'rookie_header_image_style_options', array( $this, 'header_image_style_options' ) );
		add_filter( 'rookie_header_area_sections', array( $this, 'header_area_sections' ) );
		add_filter( 'rookie_customizer_vars', array( $this, 'customizer_vars' ) );
		add_filter( 'sportspress_header_sponsors_selector', array( $this, 'header_sponsors' ), 20 );
		add_filter( 'sportspress_color_schemes', array( $this, 'color_schemes' ) );
		add_action( 'after_setup_theme', array( $this, 'updater' ) );

		// Default colors
		add_filter( 'rookie_custom_background_args', array( $this, 'background' ) );
		add_filter( 'rookie_default_header_text_color', array( $this, 'default_header_text_color' ) );
		add_filter( 'rookie_default_content_color', array( $this, 'default_content_color' ) );
		add_filter( 'rookie_default_primary_color', array( $this, 'default_primary_color' ) );
		add_filter( 'rookie_default_link_color', array( $this, 'default_link_color' ) );
		add_filter( 'rookie_default_text_color', array( $this, 'default_text_color' ) );
		add_filter( 'rookie_default_background_color', array( $this, 'default_background_color' ) );
		add_filter( 'rookie_default_heading_color', array( $this, 'default_heading_color' ) );
	}

	/**
	 * Define ThemeBoy Constants.
	 */
	private function define_constants() {
		define( 'THEMEBOY_FILE', __FILE__ );
		define( 'THEMEBOY_VERSION', $this->version );
		define( 'THEMEBOY_SLUG', $this->slug );
		define( 'THEMEBOY_NAME', $this->name );
	}

	/**
	 * Include plugins.
	 */
	private function include_plugins() {
		include_once get_template_directory() . '/plugins/mega-slider/mega-slider.php';
		include_once get_template_directory() . '/plugins/news-widget/news-widget.php';
		include_once get_template_directory() . '/plugins/social-sidebar/social-sidebar.php';
	}

	/**
	 * Theme starter content.
	 */
	public function theme_starter_content( $content = array() ) {
		$content['widgets']['sidebar-1'][0][1]['show_league'] = 1;
		return $content;
	}

	public function customize_register( $wp_customize ) {
	    /*
	     * Footer Section
	     */
	    $wp_customize->add_section( 'rookie_footer' , array(
	        'title'      => __( 'Footer', 'rookie' ),
	    ) );

	    /**
	     * Copyright
	     */
	    $wp_customize->add_setting( 'themeboy[footer_copyright]', array(
	        'default'       => '',
	        'sanitize_callback' => 'sanitize_text_field',
	        'capability'    => 'edit_theme_options',
	        'type'          => 'option',
	    ) );

	    $wp_customize->add_control( 'themeboy_footer_copyright', array(
	        'label'     => __('Copyright', 'rookie'),
	        'section'   => 'rookie_footer',
	        'settings'  => 'themeboy[footer_copyright]',
	        'input_attrs' => array(
	        	'placeholder' => sprintf( _x( '&copy; %1$s %2$s', 'copyright info', 'rookie' ), date( 'Y' ), get_bloginfo( 'name' ) ),
	        ),
	    ) );

	    /**
	     * Credit
	     */
	    $wp_customize->add_setting( 'themeboy[footer_credit]', array(
	        'default'       => '',
	        'sanitize_callback' => 'sanitize_text_field',
	        'capability'    => 'edit_theme_options',
	        'type'          => 'option',
	    ) );

	    $wp_customize->add_control( 'themeboy_footer_credit', array(
	        'label'     => __('Credit', 'rookie'),
	        'section'   => 'rookie_footer',
	        'settings'  => 'themeboy[footer_credit]',
	        'input_attrs' => array(
	        	'placeholder' => sprintf( __( 'Designed by %s', 'rookie' ), 'ThemeBoy' ),
	        ),
	    ) );

	    /**
	     * Link URL
	     */
	    $wp_customize->add_setting( 'themeboy[footer_link_url]', array(
	        'default'       => '',
	        'sanitize_callback' => 'esc_url',
	        'capability'    => 'edit_theme_options',
	        'type'          => 'option',
	    ) );

	    $wp_customize->add_control( 'themeboy_footer_link_url', array(
	        'label'     => __('Link URL', 'rookie'),
	        'section'   => 'rookie_footer',
	        'settings'  => 'themeboy[footer_link_url]',
	        'input_attrs' => array(
	        	'placeholder' => 'http://themeboy.com/',
	        ),
	    ) );
    
    	$wp_customize->remove_setting( 'themeboy[content_background]' );
	}
	
	public function footer_copyright( $copyright ) {
		$options = (array) get_option( 'themeboy', array() );

		// Return if not customized
		if ( ! isset( $options['footer_copyright'] ) || '' == $options['footer_copyright'] ) {
			return $copyright;
		} else {
			return $options['footer_copyright'];
		}
	}
	
	public function footer_credit( $credit ) {
		$options = (array) get_option( 'themeboy', array() );

		// Return if not customized
		if ( ( ! isset( $options['footer_credit'] ) || '' == $options['footer_credit'] ) && ( ! isset( $options['footer_link_url'] ) || '' == $options['footer_link_url'] ) ) {
			return $credit;
		} else {
			$text = sprintf( __( 'Designed by %s', 'rookie' ), 'ThemeBoy' );
			$url = 'http://themeboy.com/';
			
			if ( isset( $options['footer_credit'] ) && '' !== $options['footer_credit'] ) {
				$text = $options['footer_credit'];
			}
			
			if ( isset( $options['footer_link_url'] ) && '' !== $options['footer_link_url'] ) {
				$url = $options['footer_link_url'];
			}
			
			return '<a href="' . $url . '">' . $text . '</a>';
		}
	}

	public function header_image_style_options( $options ) {
		$options = array(
			'image' => __( 'Image', 'rookie' ),
		);
		return $options;
	}
    

	public function header_area_sections() {
		$sections = array( 'widgets', 'banner', 'branding', 'menu' );
		return $sections;
	}
	
	public function header_sponsors() {
		return '.site-header .nav-menu';
	}

	public function scripts() {
		// Remove default Rookie styles
		remove_action( 'wp_print_scripts', 'rookie_custom_colors', 30 );
		wp_dequeue_style( 'rookie-lato' );

		// Load Titillium Web font
		wp_enqueue_style( 'marquee-titillium-web', add_query_arg( 'family', 'Titillium+Web:300,600', "//fonts.googleapis.com/css", array(), null ) );

		// Apply custom colors
		add_action( 'wp_print_scripts', array( $this, 'custom_colors' ), 30 );
	}

	public function customizer_vars( $vars ) {	
		$vars['content_width_selector'] = '.site-branding, .site-content, .site-footer .footer-area, .site-info, .main-navigation';
		return $vars;
	}

	public function custom_colors() {
		$colors = (array) get_option( 'themeboy', array() );
		$colors = array_map( 'esc_attr', $colors );
		
		// Get layout options
		if ( empty( $colors['content_width'] ) ) {
			$width = 1000;
		} else {
			$width = rookie_sanitize_content_width( $colors['content_width'] );
		}

		global $content_width;

		if ( empty( $colors['sidebar'] ) ) {
			$sidebar = '';
		} else {
			$sidebar = $colors['sidebar'];
		}

		if ( 'no' == $sidebar || is_page_template( 'template-fullwidth.php' ) ) {
			$content_width = $width - 30;
		} elseif ( 'double' === $sidebar )  {
			$content_width = $width * .50 - 30;
		} else {
			$content_width = $width * .66 - 30;
		}

		?>
		<style type="text/css"> /* Marquee Custom Layout */
		@media screen and (min-width: 1025px) {
			.site-branding, .site-content, .site-footer .footer-area, .site-info {
				width: <?php echo $width; ?>px; }
			.main-navigation {
				width: <?php echo $width - 30; ?>px; }
		}
		</style>
		<?php

		// Return if not customized
		if ( ! isset( $colors['customize'] ) ) {
			$enabled = get_option( 'sportspress_enable_frontend_css', 'no' );
			if ( 'yes' !== $enabled ) return;
		} elseif ( ! $colors['customize'] ) {
			return;
		}
		
		$colors = array_map( 'esc_attr', $colors );
		$colors['content_background'] = '#' . get_background_color();

		// Defaults
		if ( empty( $colors['primary'] ) ) $colors['primary'] = '#2b353e';
		if ( empty( $colors['background'] ) ) $colors['background'] = '#f4f4f4';
		if ( empty( $colors['content'] ) ) $colors['content'] = '#000000';
		if ( empty( $colors['text'] ) ) $colors['text'] = '#666666';
		if ( empty( $colors['heading'] ) ) $colors['heading'] = '#ffffff';
		if ( empty( $colors['link'] ) ) $colors['link'] = '#00a69c';
		if ( empty( $colors['content_background'] ) ) $colors['content_background'] = '#ffffff';

		// Calculate colors
		$colors['primary_alt'] = rookie_hex_lighter( $colors['primary'], 15, true );
		$colors['indicator'] = rookie_hex_lighter( $colors['primary'], 50, true );
		$colors['link_hover'] = rookie_hex_darker( $colors['link'], 35, true );
		
		if ( rookie_hex_brightness( $colors['background'] ) >= 128 ) {
			$colors['background_alt'] = rookie_hex_darker( $colors['background'], 12, true );
			$colors['background_hard'] = '#000000';
			$colors['border'] = rookie_hex_darker( $colors['background'], 63, true );
			$colors['pad'] = rookie_hex_darker( $colors['background'], 31, true );
		} else {
			$colors['background_alt'] = rookie_hex_lighter( $colors['background'], 12, true );
			$colors['background_hard'] = '#ffffff';
			$colors['border'] = rookie_hex_lighter( $colors['background'], 63, true );
			$colors['pad'] = rookie_hex_lighter( $colors['background'], 31, true );
		}
		
		if ( rookie_hex_brightness( $colors['heading'] ) >= 128 ) {
			$colors['heading_contrast'] = '#000000';
		} else {
			$colors['heading_contrast'] = '#ffffff';
		}

		?>
		<style type="text/css"> /* Frontend CSS */
		blockquote:before,
		q:before {
			color: <?php echo $colors['primary']; ?>; }
		button,
		input[type="button"],
		input[type="reset"],
		input[type="submit"],
		.site-header,
		.widget_calendar caption,
		.sp-table-caption,
		.sp-template-countdown,
		.sp-scoreboard-wrapper,
		.sp-statistic-bar-fill,
		.nav-links a {
			background-color: <?php echo $colors['primary']; ?>; }
		button:hover,
		input[type="button"]:hover,
		input[type="reset"]:hover,
		input[type="submit"]:hover,
		.nav-links a:hover {
			background-color: <?php echo $colors['primary_alt']; ?>; }
		.main-navigation .current-menu-item:after,
		.main-navigation .current-menu-parent:after,
		.main-navigation .current-menu-ancestor:after,
		.main-navigation .current_page_item:after,
		.main-navigation .current_page_parent:after,
		.main-navigation .current_page_ancestor:after {
			background-color: <?php echo $colors['indicator']; ?>; }
		.header-area,
		.sp-event-blocks .sp-event-results a:hover .sp-result,
		.sp-event-blocks .alternate .sp-event-results .sp-result,
		.mega-slider__row {
			background-color: <?php echo $colors['content_background']; ?>; }
		table,
		.sp-event-logos .sp-team-logo .sp-team-result,
		.sp-template-countdown h5,
		.sp-player-details,
		.sp-staff-details,
		.sp-scoreboard .sp-highlight,
		.mega-slider__row:hover {
			background-color: <?php echo $colors['background']; ?>; }
		table tbody tr:nth-child(2n+1),
		.sp-data-table tbody tr.odd,
		.site-footer,
		.sp-event-blocks .sp-event-results .sp-result,
		.sp-event-blocks .alternate .sp-event-results a:hover .sp-result,
		.sp-template-countdown .event-venue,
		.mega-slider__row:nth-child(2n+1) {
			background-color: <?php echo $colors['background_alt']; ?>; }
		table,
		td,
		.site-footer,
		.sp-event-blocks .sp-event-results .sp-result,
		.mega-slider__sidebar {
			border-color: <?php echo $colors['border']; ?>; }
		.widget_calendar tbody .pad,
		.sp-template-event-calendar tbody .pad {
			background-color: <?php echo $colors['pad']; ?>; }
		td,
		blockquote,
		q,
		.widget,
		.site-info,
		.sp-template-event-blocks .sp-event-date a,
		.news-widget__post__date,
		.news-widget__post__date a,
		.entry-meta a,
		.posted-on a {
			color: <?php echo $colors['text']; ?>; }
		body,
		table tfoot tr td,
		.widget-title,
		.widget_news_widget,
		.entry-header,
		.sp-highlight,
		.sp-template-event-blocks .sp-event-title a,
		.sp-template-event-blocks .sp-event-results a,
		.sp-event-blocks .sp-event-results .sp-result,
		.sp-template-countdown h5 {
			color: <?php echo $colors['content']; ?>; }
		a,
		.mega-slider__row__link {
			color: <?php echo $colors['link']; ?>; }
		.widget_calendar #today,
		.sp-template-event-calendar #today,
		.gallery dd strong,
		.single-sp_player .entry-header .entry-title strong,
		.single-sp_staff .entry-header .entry-title strong {
			background-color: <?php echo $colors['link']; ?>; }
		.widget_calendar #today,
		.sp-template-event-calendar #today {
			border-color: <?php echo $colors['link']; ?>; }
		.sp-tab-menu-item-active a {
			border-bottom-color: <?php echo $colors['link']; ?>; }
		a:hover,
		.mega-slider__row__link:hover {
			color: <?php echo $colors['link_hover']; ?>; }
		th,
		table thead,
		button,
		input[type="button"],
		input[type="reset"],
		input[type="submit"],
		.main-navigation,
		.main-navigation a,
		.widget_calendar caption,
		.sp-table-caption,
		.widget_calendar #prev a:before,
		.sp-template-event-calendar #prev a:before,
		.widget_calendar #next a:before,
		.sp-template-event-calendar #next a:before,
		.widget_calendar #today,
		.sp-template-event-calendar #today,
		.widget_calendar #today a,
		.sp-template-event-calendar #today a,
		.gallery dd,
		.gallery dd strong,
		.sp-template-countdown,
		.single-sp_player .entry-header .entry-title strong,
		.single-sp_staff .entry-header .entry-title strong,
		.sp-scoreboard-nav,
		.mega-slider__row--active,
		.mega-slider__row--active:nth-child(2n+1),
		.mega-slider__row--active:hover,
		.mega-slider__row--active .mega-slider__row__date,
		.mega-slider__row--active .mega-slider__row__link,
		.nav-links a {
			color: <?php echo $colors['heading']; ?>; }
		th,
		table thead,
		.main-navigation,
		.gallery dd,
		.sp-template-countdown .event-name,
		.site-widgets .sp-scoreboard-wrapper,
		.sp-statistic-bar {
			background-color: <?php echo $colors['heading_contrast']; ?>; }
		th,
		table thead {
			border-color: <?php echo $colors['heading_contrast']; ?>; }
		.mega-slider__row--active,
		.mega-slider__row--active:nth-child(2n+1),
		.mega-slider__row--active:hover {
			background-color: <?php echo $colors['primary']; ?>; }
			
		@media screen and (min-width: 601px) {
			.main-navigation,
			.main-navigation ul ul,
			.main-navigation li:hover {
				background-color: <?php echo $colors['content']; ?>; }
		}

		<?php do_action( 'sportspress_frontend_css', $colors ); ?>

		</style>
		<?php
	}

	public function color_schemes( $color_schemes ) {
		$color_schemes['Marquee'] = array( '0f8bca', 'ffffff', '666666', 'ffffff', '0f8bca' );
		return $color_schemes;
	}

	public function updater() {
		require_once( 'updater/themeboy-updater.php' );
	}

	public function background( $args ) {
		$args['default-color'] = 'f9f9f9';
		return $args;
	}

	public function default_header_text_color() {
		return '#555555';
	}

	public function default_content_color() {
		return '#000000';
	}

	public function default_primary_color() {
		return '#0f8bca';
	}

	public function default_link_color() {
		return '#0f8bca';
	}

	public function default_text_color() {
		return '#666666';
	}

	public function default_background_color() {
		return '#f9f9f9';
	}

	public function default_heading_color() {
		return '#ffffff';
	}
}

new ThemeBoy_Marquee();

endif;

require_once( 'framework.php' );


/*
 * Function creates post duplicate as a draft and redirects then to the edit post screen
 */
function rd_duplicate_post_as_draft(){
	global $wpdb;
	if (! ( isset( $_GET['post']) || isset( $_POST['post'])  || ( isset($_REQUEST['action']) && 'rd_duplicate_post_as_draft' == $_REQUEST['action'] ) ) ) {
		wp_die('No post to duplicate has been supplied!');
	}
 
	/*
	 * Nonce verification
	 */
	if ( !isset( $_GET['duplicate_nonce'] ) || !wp_verify_nonce( $_GET['duplicate_nonce'], basename( __FILE__ ) ) )
		return;
 
	/*
	 * get the original post id
	 */
	$post_id = (isset($_GET['post']) ? absint( $_GET['post'] ) : absint( $_POST['post'] ) );
	/*
	 * and all the original post data then
	 */
	$post = get_post( $post_id );
 
	/*
	 * if you don't want current user to be the new post author,
	 * then change next couple of lines to this: $new_post_author = $post->post_author;
	 */
	$current_user = wp_get_current_user();
	$new_post_author = $current_user->ID;
 
	/*
	 * if post data exists, create the post duplicate
	 */
	if (isset( $post ) && $post != null) {
 
		/*
		 * new post data array
		 */
		$args = array(
			'comment_status' => $post->comment_status,
			'ping_status'    => $post->ping_status,
			'post_author'    => $new_post_author,
			'post_content'   => $post->post_content,
			'post_excerpt'   => $post->post_excerpt,
			'post_name'      => $post->post_name,
			'post_parent'    => $post->post_parent,
			'post_password'  => $post->post_password,
			'post_status'    => 'draft',
			'post_title'     => $post->post_title,
			'post_type'      => $post->post_type,
			'to_ping'        => $post->to_ping,
			'menu_order'     => $post->menu_order
		);
 
		/*
		 * insert the post by wp_insert_post() function
		 */
		$new_post_id = wp_insert_post( $args );
 
		/*
		 * get all current post terms ad set them to the new post draft
		 */
		$taxonomies = get_object_taxonomies($post->post_type); // returns array of taxonomy names for post type, ex array("category", "post_tag");
		foreach ($taxonomies as $taxonomy) {
			$post_terms = wp_get_object_terms($post_id, $taxonomy, array('fields' => 'slugs'));
			wp_set_object_terms($new_post_id, $post_terms, $taxonomy, false);
		}
 
		/*
		 * duplicate all post meta just in two SQL queries
		 */
		$post_meta_infos = $wpdb->get_results("SELECT meta_key, meta_value FROM $wpdb->postmeta WHERE post_id=$post_id");
		if (count($post_meta_infos)!=0) {
			$sql_query = "INSERT INTO $wpdb->postmeta (post_id, meta_key, meta_value) ";
			foreach ($post_meta_infos as $meta_info) {
				$meta_key = $meta_info->meta_key;
				if( $meta_key == '_wp_old_slug' ) continue;
				$meta_value = addslashes($meta_info->meta_value);
				$sql_query_sel[]= "SELECT $new_post_id, '$meta_key', '$meta_value'";
			}
			$sql_query.= implode(" UNION ALL ", $sql_query_sel);
			$wpdb->query($sql_query);
		}
 
 
		/*
		 * finally, redirect to the edit post screen for the new draft
		 */
		wp_redirect( admin_url( 'post.php?action=edit&post=' . $new_post_id ) );
		exit;
	} else {
		wp_die('Post creation failed, could not find original post: ' . $post_id);
	}
}
add_action( 'admin_action_rd_duplicate_post_as_draft', 'rd_duplicate_post_as_draft' );
 
/*
 * Add the duplicate link to action list for post_row_actions
 */
function rd_duplicate_post_link( $actions, $post ) {
	if (current_user_can('edit_posts')) {
		$actions['duplicate'] = '<a href="' . wp_nonce_url('admin.php?action=rd_duplicate_post_as_draft&post=' . $post->ID, basename(__FILE__), 'duplicate_nonce' ) . '" title="Duplicate this item" rel="permalink">Duplicate</a>';
	}
	return $actions;
}
 
add_filter( 'post_row_actions', 'rd_duplicate_post_link', 10, 2 );
add_filter('page_row_actions', 'rd_duplicate_post_link', 10, 2);

// change page link to display published pages only
function wcs_change_admin_page_link() {
    global $submenu;
    $submenu['edit.php?post_type=page'][5][2] = 'edit.php?post_type=page&post_status=publish';
}
add_action( 'admin_menu', 'wcs_change_admin_page_link' );

// change post link to display published posts only
function wcs_change_admin_post_link() {
    global $submenu;
    $submenu['edit.php'][5][2] = 'edit.php?post_status=publish';
}
add_action( 'admin_menu', 'wcs_change_admin_post_link' );

