<?php
/*
Plugin Name: BM Content Builder
Plugin URI: https://1.envato.market/4Mjnr
Description: The BM Content Builder allows you to customize your content in live mode and see the results instantly. It works with the SEA theme, the ART theme version 3. 
Author: SeaTheme
Author URI: http://seatheme.net 
Text Domain: bm-builder
Domain Path: /languages/
Version:  3.5.0
License:  You should have purchased a license from Themeforest, author:bwsm
*/

// don't load directly
if ( ! defined( 'ABSPATH' ) ) {
	die( '-1' );
}
/**
 * Current content builder version
 */
if ( ! defined( 'UX_CB_VERSION' ) ) {
	/**
	 *
	 */
	define( 'UX_CB_VERSION', '1.0.0' );
}

/**
 * Main UX_ContentBuilder Class.
 *
 * @class UX_ContentBuilder
 * @version	1.0.0
 */
class UX_ContentBuilder {
	/**
	 * List of paths.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	private $paths = array();
	
	/**
	 * Core singleton class
	 * @var self - pattern realization
	 */
	private static $_instance;
	
	/**
	 * File name for components manifest file.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	private $components_manifest = 'components.json';
	
	/**
	 * @var string
	 */
	private $plugin_name = 'ux-content-builder/ux-content-builder.php';
	
	/**
	 * Constructor loads API functions, defines paths and adds required wp actions
	 *
	 * @since  1.0.0
	 */
	private function __construct() {
		$dir = dirname( __FILE__ );
		/**
		 * Define path settings for content builder.
		 *
		 * APP_ROOT        - plugin directory.
		 * WP_ROOT         - WP application root directory.
		 * APP_DIR         - plugin directory name.
		 * CONFIG_DIR      - configuration directory.
		 * ASSETS_DIR      - asset directory full path.
		 * ASSETS_DIR_NAME - directory name for assets. Used from urls creating.
		 * CORE_DIR        - classes directory for core content builder files.
		 * HELPERS_DIR     - directory with helpers functions files.
		 * SHORTCODES_DIR  - shortcodes classes.
		 * SETTINGS_DIR    - main dashboard settings classes.
		 * TEMPLATES_DIR   - directory where all html templates are hold.
		 * EDITORS_DIR     - editors for the post contents
		 * PARAMS_DIR      - complex params for shortcodes editor form.
		 * UPDATERS_DIR    - automatic notifications and updating classes.
		 */
		$this->setPaths( array(
			'APP_ROOT' => $dir,
			'WP_ROOT' => preg_replace( '/$\//', '', ABSPATH ),
			'APP_DIR' => basename( plugin_basename( $dir ) ),
			'CONFIG_DIR' => $dir . '/config',
			'ASSETS_DIR' => $dir . '/assets',
			'PLACEHOLD_DIR' => $dir . '/placehold',
			'PLACEHOLD_DIR_NAME' => 'placehold',
			'ASSETS_DIR_NAME' => 'assets',
			'AUTOLOAD_DIR' => $dir . '/include/autoload',
			'CORE_DIR' => $dir . '/include/classes/core',
			'HELPERS_DIR' => $dir . '/include/helpers',
			'SHORTCODES_DIR' => $dir . '/include/classes/shortcodes',
			'SETTINGS_DIR' => $dir . '/include/classes/settings',
			'TEMPLATES_DIR' => $dir . '/include/templates',
			'EDITORS_DIR' => $dir . '/include/classes/editors',
			'PARAMS_DIR' => $dir . '/include/params',
			'UPDATERS_DIR' => $dir . '/include/classes/updaters',
			'VENDORS_DIR' => $dir . '/include/classes/vendors',
		) );
		
		// Load API
		require_once $this->path( 'HELPERS_DIR', 'helpers.php' );
		require_once $this->path( 'HELPERS_DIR', 'helpers_interface.php' );
		require_once $this->path( 'HELPERS_DIR', 'helpers_factory.php' );
		require_once $this->path( 'HELPERS_DIR', 'helpers_embed.php' );
		require_once $this->path( 'HELPERS_DIR', 'helpers_style.php' );
		require_once $this->path( 'HELPERS_DIR', 'helpers_import.php' );
		require_once $this->path( 'HELPERS_DIR', 'helpers_fonts.php' );
		//require_once $this->path( 'HELPERS_DIR', 'helpers_bmtools.php' );
		require_once $this->path( 'SETTINGS_DIR', 'fields.php' );
		
		// Add hooks
		add_action( 'plugins_loaded', array( $this, 'pluginsLoaded' ), 9 );
		add_action( 'init', array( $this, 'init' ), 9 );
		add_action( 'load-post.php', array( &$this, 'backendEnqueueScripts' ) );
		add_action( 'load-post-new.php', array( &$this, 'backendEnqueueScripts' ) );
		add_action( 'wp_enqueue_scripts', array( &$this, 'frontendEnqueueScripts') );
		add_action( 'after_setup_theme', array( $this, 'setupEnvironment' ) );
		add_action( 'add_meta_boxes', array( $this, 'registerPostMetaBox' ), 10, 2 );
		$this->setPluginName( $this->path( 'APP_DIR', 'ux-content-builder.php' ) );
		register_activation_hook( __FILE__, array( $this, 'activationHook' ) );
		
		//$GLOBALS['toolsOptionConfig'] = ux_cb_tools_option_config();
		$GLOBALS['ux_cb_module_fields'] = new UX_CB_Module_Fields();
		$GLOBALS['ux_cb_module_theme_color'] = ux_cb_module_theme_color();
		
	}
	
	/**
	 * Get the instane of UX_ContentBuilder
	 *
	 * @return self
	 */
	public static function getInstance() {
		if ( ! ( self::$_instance instanceof self ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}
	
	/**
	 * Load required components to enable useful functionality.
	 *
	 * @access public
	 * @since 1.0.0
	 */
	public function loadComponents() {
		$manifest_file = apply_filters( 'ux_cb_autoload_components_manifest_file', ux_cb_path_dir( 'AUTOLOAD_DIR', $this->components_manifest ) );
		if ( is_file( $manifest_file ) ) {
			ob_start();
			require_once $manifest_file;
			$data = ob_get_clean();
			if ( $data ) {
				$components = (array) json_decode( $data );
				$components = apply_filters( 'ux_cb_autoload_components_list', $components );
				$dir = ux_cb_path_dir( 'AUTOLOAD_DIR' );
				foreach ( $components as $component => $description ) {
					$component_path = $dir . '/' . $component;
					if ( false === strpos( $component_path, '*' ) ) {
						require_once $component_path;
					} else {
						$components_paths = glob( $component_path );
						if ( is_array( $components_paths ) ) {
							foreach ( $components_paths as $path ) {
								if ( false === strpos( $path, '*' ) ) {
									require_once $path;
								}
							}
						}
					}
				}
			}
		}
	}
	
	
	/**
	 * Callback function for WP init action hook. Sets CB mode and loads required objects.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return void
	 */
	
	public function setupEnvironment() {
		add_image_size('ux-standard-thumb-small', 350, 9999);
		add_image_size('ux-standard-thumb', 650, 9999);
		add_image_size('ux-standard-thumb-medium', 1000, 9999);
		add_image_size('ux-standard-thumb-big', 2000, 9999); 
	}
	
	/**
	 * Callback function WP plugin_loaded action hook. Loads locale
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function pluginsLoaded() {
		// Setup locale
		do_action( 'ux_cb_plugins_loaded' );
		load_plugin_textdomain( 'bm-builder', false, $this->path( 'APP_DIR', 'languages' ) );
	}
	
	/**
	 * Callback function for WP init action hook. Sets CB mode and loads required objects.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return void
	 */
	public function init() {
		do_action( 'ux_cb_before_init' );
		/**
		 * Set version of content builder if required.
		 */
		$this->setVersion();
		
		/**
		 * Set register javascript and css if required.
		 */
		$this->registerJavascript();
		$this->registerCss();
		
		/**
		 * Set register post type.
		 */
		$this->registerPostType();
		
		do_action( 'ux_cb_after_init' );
	}
	
	/**
	 * Register javascript libs.
	 *
	 * Calls wp_register_script for required css libraries files.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function registerJavascript() {
		//backend
		wp_register_script( 'bootstrap', ux_cb_asset_url( 'js/bootstrap.min.js' ), array( 'jquery' ), '3.3.5', true );
		wp_register_script( 'ux-cb-summernote-script', ux_cb_asset_url( 'js/summernote.js' ), array( 'jquery' ), '0.8.3', true );
		wp_register_script( 'ux-cb-backend-script', ux_cb_asset_url( 'js/backend-script.js' ), array( 'jquery' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-backend-tools', ux_cb_asset_url( 'js/backend-tools.js' ), array( 'jquery' ), UX_CB_VERSION, true );
		
		//frontend
		wp_register_script( 'lazyload', ux_cb_asset_url( 'js/lazyload.min.js' ), array( 'jquery' ), '10.19.0', true );
		wp_register_script( 'waypoints', ux_cb_asset_url( 'js/jquery.waypoints.min.js' ), array('jquery'), '4.0.1', true);
		wp_register_script( 'photoswipe-min', ux_cb_asset_url( 'js/photoswipe.min.js' ), array( 'jquery' ), '4.1.2', true );
		wp_register_script( 'photoswipe-ui-default-min', ux_cb_asset_url( 'js/photoswipe-ui-default.min.js' ), array('jquery'), '4.1.2', true);

		//modules
		wp_register_script( 'ux-cb-modules-global', ux_cb_asset_url( 'js/modules-global.js' ), array( 'jquery' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-modules-grid', ux_cb_asset_url( 'js//mod/grid.js' ), array( 'ux-cb-modules-global' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-modules-masonry-grid', ux_cb_asset_url( 'js//mod/masonry-grid.js' ), array( 'ux-cb-modules-global' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-modules-contact-form', ux_cb_asset_url( 'js//mod/contact-form.js' ), array( 'ux-cb-modules-global' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-modules-embed', ux_cb_asset_url( 'js//mod/embed.js' ), array( 'ux-cb-modules-global' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-modules-single-image', ux_cb_asset_url( 'js//mod/single-image.js' ), array( 'ux-cb-modules-global' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-modules-slider', ux_cb_asset_url( 'js//mod/slider.js' ), array( 'ux-cb-modules-global' ), UX_CB_VERSION, true );
		wp_register_script( 'ux-cb-modules-text', ux_cb_asset_url( 'js//mod/text.js' ), array( 'ux-cb-modules-global' ), UX_CB_VERSION, true );
		
		//plugins
		wp_register_script( 'gridstack', ux_cb_asset_url( 'js/gridstack.js' ), array( 'jquery' ), '0.3.0', true );
		wp_register_script( 'isotope', ux_cb_asset_url( 'js/isotope.pkgd.min.js' ), array('jquery'), '3.0.6', true);
		wp_register_script( 'packery-mode', ux_cb_asset_url( 'js/packery-mode.pkgd.min.js' ), array('jquery'), '2.0.0', true);
		wp_register_script( 'owl.carousel', ux_cb_asset_url( 'js/owl.carousel.min.js' ), array('jquery'), '2.3.4', true);
		
		/**
		 * @since 1.0.0
		 */
		do_action( 'ux_cb_base_register_front_js' );
	}
	
	/**
	 * Register css libs.
	 *
	 * Calls wp_register_style for required css libraries files.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function registerCss() {
		//backend
		wp_register_style( 'ux-cb-summernote-style', ux_cb_asset_url( 'css/summernote.css' ), array(), '0.8.3', 'all' );
		wp_register_style( 'ux-cb-backend-style', ux_cb_asset_url( 'css/backend-style.css' ), array(), UX_CB_VERSION, 'all' );
		wp_register_style( 'ux-cb-backend-tools', ux_cb_asset_url( 'css/backend-tools.css' ), array(), UX_CB_VERSION, 'all' );

		//photoswipe
		wp_register_style( 'photoswipe', ux_cb_asset_url( 'css/photoswipe.css' ), array(), '4.1.2', 'all' );
		
		//bootstrap
		wp_register_style( 'bootstrap', ux_cb_asset_url( 'css/bootstrap.css' ), array(), '3.3.7', 'all' );

		//modules
		wp_register_style( 'ux-cb-modules-style', ux_cb_asset_url( 'css/modules-style.css' ), array(), UX_CB_VERSION, 'all' );
		
		//plugins
		wp_register_style( 'gridstack', ux_cb_asset_url( 'css/gridstack.min.css' ), array(), '1.0.0', 'all' );
		wp_register_style( 'owl-carousel', ux_cb_asset_url( 'css/owl.carousel.css' ), array(), '2.3.0', 'all' );
		
		/**
		 * @since 1.0.0
		 */
		do_action( 'ux_cb_base_register_front_css' );
	}
	
	/**
	 * Register post type.
	 */
	public function registerPostType() {
		
		require_once $this->path( 'AUTOLOAD_DIR', 'ux-cb-post-type/ux-portfolio.php' );
		require_once $this->path( 'AUTOLOAD_DIR', 'ux-cb-post-type/ux-attachment.php' );
		require_once $this->path( 'AUTOLOAD_DIR', 'ux-cb-post-type/ux-modules.php' );
		
		register_post_type('ux-cb-template', array('label' => __('Module Template (CB)','bm-builder'), 'show_ui' => false));
		
	}
	
	/**
	 * Set register post meta box.
	 */
	public function registerPostMetaBox( $post_type, $post ) {
		global $ux_theme_post_meta_fields, $ux_theme_fn_prefix;
		
		if ( $ux_theme_post_meta_fields ) {
			$postMetaFields = $ux_theme_post_meta_fields;
			if ( $post_type != 'page' ) {
				if ( function_exists( 'use_block_editor_for_post' ) ) {
					if ( use_block_editor_for_post( $post ) ) {
						if ( ! empty( $postMetaFields[ $post_type ] ) ) {
							$postMetaPosttype = $postMetaFields[ $post_type ];
							
							foreach ( $postMetaPosttype as $num => $option ) {
								$option = wp_parse_args( $option, array(
									'title' => false,
									'id' => false
								) );
								
								add_meta_box( $option['id'], $option['title'], $ux_theme_fn_prefix. '_theme_meta_box_callback',
									$post_type, 'normal', 'high',
									array(
										'__block_editor_compatible_meta_box' => true,
										'__back_compat_meta_box' => false,
										'option' => $option
									)
								);
								
							}
						}
					}
				}
			}
			
			if ( $post_type == 'page' ) {
				if ( function_exists( 'use_block_editor_for_post' ) ) {
					if ( use_block_editor_for_post( $post ) ) {
						add_meta_box( 'ux-theme-meta-box-page-option-wp5', esc_html__( 'Page Options', 'bm-builder' ), $ux_theme_fn_prefix. '_edit_form_after_title',
							$post_type, 'normal', 'high',
							array(
								'__block_editor_compatible_meta_box' => true,
								'__back_compat_meta_box' => false
							)
						);	
					}
				}
			}
		}
	}
	
	
	/**
	 * Enqueue required javascript libraries and css files.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function backendEnqueueScripts() {
		wp_enqueue_script( 'jquery-ui-droppable' );
		wp_enqueue_script( 'jquery-ui-draggable' );
		wp_enqueue_script( 'jquery-ui-resizable' );
		wp_enqueue_script( 'wp-color-picker');
		wp_enqueue_script( 'underscore' );
		wp_enqueue_script( 'waypoints' );
		wp_enqueue_script( 'gridstack' );
		wp_enqueue_script( 'bootstrap' );
		wp_enqueue_script( 'isotope' );
		wp_enqueue_script( 'packery-mode' );
		wp_enqueue_script( 'ux-cb-summernote-script' );
		wp_enqueue_script( 'owl.carousel' );
		wp_enqueue_script( 'ux-cb-backend-script' );
		
		wp_enqueue_style( 'wp-color-picker' ); 
		wp_enqueue_style( 'ux-cb-summernote-style' );
		wp_enqueue_style( 'ux-cb-backend-style' );
		wp_enqueue_style( 'gridstack' );
		wp_enqueue_style( 'owl-carousel' );
		wp_enqueue_style( 'ux-cb-modules-style' );
		wp_add_inline_style( 'ux-cb-modules-style', ux_cb_module_inline_style() );
		
		/**
		 * @since 1.0.0
		 */
		do_action( 'ux_cb_backend_enqueue_scripts' );
	}
	
	/**
	 * Enqueue required javascript libraries and css files.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function frontendEnqueueScripts() {
		wp_enqueue_script( 'underscore' );
		wp_enqueue_script( 'waypoints' );
		wp_enqueue_script( 'isotope' );
		wp_enqueue_script( 'packery-mode' );
		wp_enqueue_script( 'lazyload' );
		wp_enqueue_script( 'owl.carousel' );
		wp_enqueue_script( 'photoswipe-min' );
		wp_enqueue_script( 'photoswipe-ui-default-min' );
		wp_enqueue_script( 'ux-cb-modules-global' );
		
		wp_enqueue_style( 'bootstrap' );
		wp_enqueue_style( 'owl-carousel' );
		wp_enqueue_style( 'ux-cb-modules-style' );
		//wp_add_inline_style( 'ux-cb-modules-style', ux_cb_fonts_assignment() );
		
		global $post;
		if ( $post ) {
			if ( is_home() ) {
				global $wp_query;
				if ( $wp_query->is_posts_page ) {
					if ( $themeOption = get_option( 'ux_theme_option' ) ) {
						$blogListShowCB = false;
						if ( isset( $themeOption['theme_option_blog_list_show_cb'] ) ) {
							$blogListShowCB = $themeOption['theme_option_blog_list_show_cb'];
							if ( $blogListShowCB == 'true' ) {
								$blogListShowCB = true;
							}
						}
						
						if ( $blogListShowCB ) {
							$wp_version = get_bloginfo( 'version' );
							
							$postID = get_queried_object_id();
							$post = get_post( $postID );
						}
					}
				}
			}
			
			if ( $hasModules = get_post_meta( $post->ID, 'modules', true ) ) {
				if ( $hasModules && is_array( $hasModules ) ) {
					foreach ( $hasModules as $moduleUnique ) { 
						$post_id = ux_cb_get_the_module_post_id( $moduleUnique );
						$moduleID = get_post_meta( $post_id, 'module_id', true );
						wp_enqueue_script( 'ux-cb-modules-' .$moduleID );
						
						switch ( $moduleID ) {
							case 'grid':
								$source = get_post_meta( $post_id, '__ux_cb_module_grid_source', true );
								if ( $source == 'post' ) {
									wp_enqueue_script( 'ux-cb-modules-embed' );
								}
							break;
							
							case 'post-navi': wp_enqueue_script( 'ux-cb-modules-slider' ); break;
						}
					}
				}
			}
		}
		
		wp_add_inline_script( 'jquery-migrate', $this->inlineScript() );
		
		/**
		 * @since 1.0.0
		 */
		do_action( 'ux_cb_frontend_enqueue_scripts' );
	}
	
	/**
	 * Sets version of the content builder in DB as option `ux_cb_version`
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @return void
	 */
	protected function setVersion() {
		$version = get_option( 'ux_cb_version' );
		if ( ! is_string( $version ) || version_compare( $version, UX_CB_VERSION ) !== 0 ) {
			update_option( 'ux_cb_version', UX_CB_VERSION );
		}
	}
	
	public function inlineScript() {
		$output = '';
		
		$output .= 'var photoSwipeLocalize = [];';
		$output .= 'photoSwipeLocalize.push({id:"facebook", label:"' .esc_attr__('Share on Facebook','bm-builder'). '", url:"https://www.facebook.com/sharer/sharer.php?u={{url}}"});';
		$output .= 'photoSwipeLocalize.push({id:"twitter", label:"' .esc_attr__('Tweet','bm-builder'). '", url:"https://twitter.com/intent/tweet?text={{text}}&url={{url}}"});';
		$output .= 'photoSwipeLocalize.push({id:"pinterest", label:"' .esc_attr__('Pin it','bm-builder'). '", url:"http://www.pinterest.com/pin/create/button/"+ "?url={{url}}&media={{image_url}}&description={{text}}"});';
		$output .= 'photoSwipeLocalize.push({id:"download", label:"' .esc_attr__('Download image','bm-builder'). '", url:"{{raw_image_url}}", download:true});';
		
		return $output;
	}
	
	/**
	 * Setter for paths
	 *
	 * @since  1.0.0
	 * @access protected
	 *
	 * @param $paths
	 */
	protected function setPaths( $paths ) {
		$this->paths = $paths;
	}
	
	/**
	 * Gets absolute path for file/directory in filesystem.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @param $name - name of path dir
	 * @param string $file - file name or directory inside path
	 *
	 * @return string
	 */
	public function path( $name, $file = '' ) {
		$path = $this->paths[ $name ] . ( strlen( $file ) > 0 ? '/' . preg_replace( '/^\//', '', $file ) : '' );

		return apply_filters( 'ux_cb_path_filter', $path );
	}
	
	/**
	 * @since 1.0.0
	 *
	 */
	public function setPluginName( $name ) {
		$this->plugin_name = $name;
	}
	
	/**
	 * Enables to add hooks in activation process.
	 * @since 1.0.0
	 *
	 * @param $networkWide
	 */
	public function activationHook( $networkWide = false ) {
		do_action( 'ux_cb_activation_hook', $networkWide );
	}

	/**
	 * Get absolute url for asset file.
	 *
	 * Assets are css, javascript, less files and images.
	 *
	 * @since 1.0.0
	 *
	 * @param $file
	 *
	 * @return string
	 */
	public function assetUrl( $file ) {
		return preg_replace( '/\s/', '%20', plugins_url( $this->path( 'ASSETS_DIR_NAME', $file ), __FILE__ ) );
	}

	/**
	 * Get absolute url for placehol file.
	 *
	 * @since 1.0.0
	 *
	 * @param $file
	 *
	 * @return string
	 */
	public function placeholdUrl( $file ) {
		$dirPath = preg_replace( '/\s/', '%20', plugins_url( $this->path( 'PLACEHOLD_DIR_NAME', $file ), __FILE__ ) );
		if ( defined( 'INS_TPL_PLACEHOLD_DIR' ) ) {
			$dirPath = get_template_directory_uri(). '/placehold/'. $file;
		}
		
		return $dirPath;
	}
	
	public function export_post_type() {
		$post_type = array( 'post', 'page', 'attachment', 'nav_menu_item' );
		
		if ( post_type_exists( 'ux-cb-modules' ) ) {
			array_push( $post_type, 'ux-cb-modules' );
		}
		
		if ( post_type_exists( 'ux-cb-template' ) ) {
			array_push( $post_type, 'ux-cb-template' );
		}
		
		if ( post_type_exists( 'ux-portfolio' ) ) {
			array_push( $post_type, 'ux-portfolio' );
		}
		
		if ( post_type_exists( 'bmslider' ) ) {
			array_push( $post_type, 'bmslider' );
		}
		
		return $post_type;
	}
}

/**
 * Main Content Builder.
 * @var UX_ContentBuilder $ux_content_builder - instance of composer management.
 * @since 1.0.0
 */
global $ux_content_builder;
if ( ! $ux_content_builder ) {
	$ux_content_builder = UX_ContentBuilder::getInstance();
	// Load components
	$ux_content_builder->loadComponents();
	
	$GLOBALS['ux_content_builder'] = $ux_content_builder;
}
?>